///=============================================================================
// RPG Maker MZ - Timed Text Display
//=============================================================================

/*:
 * @target MZ
 * @plugindesc Displays text on the screen for a specified duration.
 * @author YourName
 *
 * @help TimedTextDisplay.js
 *
 * This plugin allows you to display text on the screen for a specified duration.
 *
 * Plugin Command:
 *   Show Timed Text
 *     - Text: The text to display. Control characters are allowed.
 *     - X: X position of the text on the screen.
 *     - Y: Y position of the text on the screen.
 *     - Duration: How long (in frames) the text will be visible.
 *
 * @command showTimedText
 * @text Show Timed Text
 * @desc Displays text on the screen for a specified duration.
 *
 * @arg text
 * @type multiline_string
 * @text Text
 * @desc The text to display. Control characters are allowed.
 *
 * @arg x
 * @type number
 * @text X Position
 * @desc The X position of the text on the screen.
 * @default 0
 *
 * @arg y
 * @type number
 * @text Y Position
 * @desc The Y position of the text on the screen.
 * @default 0
 *
 * @arg duration
 * @type number
 * @text Duration
 * @desc How long (in frames) the text will be visible.
 * @default 120
 */

(() => {
    const pluginName = "TimedTextDisplay";

    PluginManager.registerCommand(pluginName, "showTimedText", args => {
        const text = String(args.text);
        const x = Number(args.x);
        const y = Number(args.y);
        const duration = Number(args.duration);

        const scene = SceneManager._scene;
        if (scene && scene instanceof Scene_Map) {
            scene.showTimedText(text, x, y, duration);
        } else {
            console.warn("Timed Text can only be displayed on the map scene.");
        }
    });

    Scene_Map.prototype.showTimedText = function(text, x, y, duration) {
        if (!this._timedTextSprites) {
            this._timedTextSprites = [];
        }

        const sprite = new Sprite(new Bitmap(Graphics.width, Graphics.height));
        sprite.bitmap.fontSize = 28;
        sprite.bitmap.drawText(text, x, y, Graphics.width - x, 36, "left");
        this.addChild(sprite);

        const startFrame = Graphics.frameCount;
        const endFrame = startFrame + duration;

        sprite.update = () => {
            if (Graphics.frameCount >= endFrame) {
                this.removeChild(sprite);
                const index = this._timedTextSprites.indexOf(sprite);
                if (index >= 0) {
                    this._timedTextSprites.splice(index, 1);
                }
            }
        };

        this._timedTextSprites.push(sprite);
    };

    const _Scene_Map_update = Scene_Map.prototype.update;
    Scene_Map.prototype.update = function() {
        _Scene_Map_update.call(this);
        if (this._timedTextSprites) {
            for (const sprite of this._timedTextSprites) {
                sprite.update();
            }
        }
    };
})();
